/*
 * Routines for updating our current fabric from a reference fabric
 */

#include <sys/time.h>

#include "db.h"
#include "libfma.h"
#include "lf_fabric.h"
#include "lf_load_fabric.h"
#include "lf_topo_map.h"
#include "lf_scheduler.h"
#include "lf_alert.h"
#include "lf_product_def.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_fabric.h"
#include "fms_notify.h"
#include "fms_fma.h"
#include "fms_switch.h"

/*
 * local prototypes
 */
static void fms_mrf_clear_mates(struct lf_fabric *fp);
static int fms_mrf_match_nics(struct lf_fabric *fp, struct lf_fabric *rfp);
static int fms_mrf_match_xbars_by_xid(struct lf_fabric *fp,
                                      struct lf_fabric *rfp);
static int fms_mrf_match_xbars_by_conns(struct lf_fabric *fp,
                                        struct lf_fabric *rfp);
static int fms_mrf_match_xbar_by_conns(struct lf_xbar *xp);
static int fms_mrf_mate_xbars(struct lf_xbar *xp,
                              struct lf_xbar *rxp, int offset);
static int fms_mrf_process_unmatched_reference_xbars(struct lf_fabric *fp,
                                                     struct lf_fabric *rfp);
static int fms_mrf_adjust_xbar_links(struct lf_fabric *fp);
static void fms_mrf_link_down(union lf_node *np, int port);
static void fms_mrf_link_up(union lf_node *np, int port);
static int fms_mrf_adjust_nic_links(struct lf_fabric *fp);
static int fms_mrf_validate_fabric(struct lf_fabric *fp);

/*
 * Do the work of update the current fabric with a fabric that we got 
 * from asking an FMA to create a map.
 * First, the NICs:
 *  We start by stepping through each NIC in the existing fabric and
 *  finding its counterpart in the new fabric.
 *  Any NICs in the new fabric but not in the old are added as existing to
 *  a bogus host and will probably get claimed by an FMA later on.
 *  Any NICs in the old fabric but not in the new just remain untouched
 *  unless a conflict arises, in which case the link is deleted.
 *
 * Next, on to the xbars:
 *  Every new xbar with an xbar ID is matched to one in the old fabric if
 *  a match can be found.  If no match is found it is just left unmatched as
 *  the xbar might have been replaced.
 *  Now, label the xbars with their closest distance to a host.
 *  Go through all old xbars in order of increasing distance and try to match
 *  to new xbars via a voting technique.
 *
 * If things get too confusing at any point during the process, the FMS
 * will just exit with a recommendation to build a real database.
 */
int
fms_update_current_fabric_with_mapped(
  struct lf_fabric *fp,
  struct lf_fabric *rfp)
{
  int rc;
  int x;

  /* reset all mate connections */
  fms_mrf_clear_mates(fp);

  /* mark all xbars in fabric as unresolved */
  for (x=0; x<fp->num_xbars; ++x) {
    FMS_XBAR(fp->xbars[x])->mrf_resolved = FALSE;
  }

  /* first, try to match up the NICs in the two fabrics */
  rc = fms_mrf_match_nics(fp, rfp);
  if (rc == -1) {
    LF_ERROR(("Unable to match NICs between current fabric and mapped fabric"));
  }

  /* Now, match up all the xbars we can using xbar IDs */
  rc = fms_mrf_match_xbars_by_xid(fp, rfp);
  if (rc == -1) {
    LF_ERROR(("Unable to match xbar IDs between current and mapped fabric"));
  }

  /* Try to match xbars by connections */
  rc = fms_mrf_match_xbars_by_conns(fp, rfp);
  if (rc == -1) {
    LF_ERROR(("Unable to match IDless xbars"));
  }

  /* Now pull in any un-matched xbars from the reference fabric */
  rc = fms_mrf_process_unmatched_reference_xbars(fp, rfp);
  if (rc == -1) {
    LF_ERROR(("Unable to process unmatched reference xbars"));
  }

  /* make sure xbar array is good */
  lf_build_xbar_array(fp);

  /* We have matched everything we can - scan through all elements and
   * make the reference fabric supercede the existing fabric
   */
  rc = fms_mrf_adjust_xbar_links(fp);
  if (rc == -1) {
    LF_ERROR(("Error adjusting xbar links"));
  }
  rc = fms_mrf_adjust_nic_links(fp);
  if (rc == -1) {
    LF_ERROR(("Error adjusting nic links"));
  }

  /* Do some validation on what we have... */
  rc = fms_mrf_validate_fabric(fp);
  if (rc == -1) {
    LF_ERROR(("Fabric validation failed after merge with reference"));
  }

  return 0;

 except:
  return -1;
}

/*
 * Clear all mate fields in primary fabric
 */
static void
fms_mrf_clear_mates(
  struct lf_fabric *fp)
{
  int x;
  int h;

  for (x=0; x<fp->num_xbars; ++x) {
    FMS_XBAR(fp->xbars[x])->mrf_mate = NULL;
  }

  for (h=0; h<fp->num_hosts; ++h) {
    struct lf_host *hp;
    int n;

    hp = fp->hosts[h];
    for (n=0; n<hp->num_nics; ++n) {
      FMS_NIC(hp->nics[n])->mrf_mate = NULL;
    }
  }
}

/*
 * Match all the NICs in the reference fabric with NICs in the current
 * fabric.  If NIC not found in current fabric, add it as owned by a bogus
 * host.
 *
 * This is also where we update dynamic fields reported by the nodes, like
 * flags and mag_id.
 */
static int
fms_mrf_match_nics(
  struct lf_fabric *fp,
  struct lf_fabric *rfp)
{
  int h;

  /* Loop over all hosts in original fabric, matching each NIC within */
  for (h=0; h<fp->num_hosts; ++h) {
    struct lf_host *hp;
    int n;

    hp = fp->hosts[h];

    for (n=0; n<hp->num_nics; ++n) {
      struct lf_nic *nicp;
      struct lf_nic *rnicp;

      /* get pointer to this NIC and see if we can find it */
      nicp = hp->nics[n];
      rnicp = lf_find_nic_by_mac(rfp, nicp->mac_addr);

      /* If found, link them together */
      if (rnicp != NULL && nicp->num_ports == rnicp->num_ports) {

	/* Update some fields from the new host */
	if (hp->fma_flags != rnicp->host->fma_flags
	    || hp->fw_type != rnicp->host->fw_type) {
	  hp->fma_flags = rnicp->host->fma_flags;
	  hp->fw_type = rnicp->host->fw_type;
	  fms_update_host_in_db(hp);
	  fms_notify(FMS_EVENT_DEBUG, "%s updated in DB", hp->hostname);
	}

	/* copy over mag_id */
	nicp->mag_id = rnicp->mag_id;

	FMS_NIC(nicp)->mrf_mate = rnicp;
	FMS_NIC(rnicp)->mrf_mate = nicp;
	fms_notify(FMS_EVENT_DEBUG, "%s:%d mated to %s:%d",
	    rnicp->host->hostname, rnicp->host_nic_id,
	    nicp->host->hostname, nicp->host_nic_id);

	/* If this host was "disconnected," it's not now */
	if (FMS_HOST(hp)->disconnected) {
	  FMS_HOST(hp)->disconnected = FALSE;
	  fms_notify(FMS_EVENT_INFO, "%s is no longer disconnected",
	      hp->hostname);
	}
      }
    }
  }

  /* Now loop through reference fabric, creating a new bogus host in the
   * primary fabric for each unmatched reference NIC in the reference fabric.
   */
  for (h=0; h<rfp->num_hosts; ++h) {
    struct lf_host *rhp;
    struct lf_nic *rnicp;

    /* sanity check */
    rhp = rfp->hosts[h];
    if (rhp->num_nics != 1) {
      LF_ERROR(("Error: Unmatched reference host has != 1 NICs!"));
    }

    /* Get to bogus host's NIC */
    rnicp = rhp->nics[0];

    /* If not mated, create a new bogus host for it */
    if (FMS_NIC(rnicp)->mrf_mate == NULL) {
      struct lf_host *nhp;
      struct lf_nic *nnicp;

      /* clone this host and add it to the main fabric */
      nhp = fms_clone_host(rhp);
      if (nhp == NULL) {
	LF_ERROR(("Error cloning host"));
      }
      lf_add_existing_host_to_fabric(fp, nhp);
      fms_add_host_to_db(nhp);

      /* get pointer to the new cloned NIC */
      nnicp = nhp->nics[0];

      /* link these two */
      FMS_NIC(nnicp)->mrf_mate = rnicp;
      FMS_NIC(rnicp)->mrf_mate = nnicp;
      fms_notify(FMS_EVENT_DEBUG, "%s:%d created",
	  rnicp->host->hostname, rnicp->host_nic_id);
    }
  }

  return 0;

 except:
  return -1;
}

/*
 * Link up any xbars with matching XIDs
 */
static int
fms_mrf_match_xbars_by_xid(
  struct lf_fabric *fp,
  struct lf_fabric *rfp)
{
  int x;

  /* loop through all the xbars and find matching xids */
  for (x=0; x<fp->num_xbars; ++x) {
    struct lf_xbar *xp;

    xp = fp->xbars[x];
    if (xp->xbar_id != 0) {
      struct lf_xbar *rxp;

      /* see if we can find a match */
      rxp = lf_find_xbar_by_id(rfp, xp->xbar_id);
      if (rxp != NULL) {

	/* Make sure this xbar is already mated */
	if (FMS_XBAR(xp)->mrf_mate != NULL ||
	    FMS_XBAR(rxp)->mrf_mate != NULL) {
	  LF_ERROR(("xbar is already mated"));
	}

	/* link 'em up */
	fms_notify(FMS_EVENT_DEBUG, "by xid: %s is really %s:%d:%d",
	  rxp->linecard->enclosure->name,
	  xp->linecard->enclosure->name,
	  lf_slot_display_no(xp->linecard),
	  xp->xbar_no);
	fms_mrf_mate_xbars(xp, rxp, 0);

	/* copy quadrant_disable value */
	xp->quadrant_disable = rxp->quadrant_disable;

        /* mark xbar as "resolved */
        FMS_XBAR(xp)->mrf_resolved = TRUE;
      }
    }
  }
  return 0;

 except:
  return -1;
}

/*
 * A new linecard was just noticed in an enclosure - see if we can replace any
 * bogus xbars
 */
void
fms_merge_new_linecard(
  struct lf_linecard *lp)
{
  struct lf_fabric *fp;
  int x;

  /* If no xbar IDs, nothing to do */
  if (lp->def->xbar_id_style == LF_XS_NONE) return;

  fp = F.fabvars->fabric;

  /* Check each xbar for a match in the fabric */
  for (x=0; x<lp->num_xbars; ++x) {
    struct lf_xbar *xp;
    struct lf_xbar *bxp;

    xp = LF_XBAR(lp->xbars[x]);

    /*
     * see if we have an xbar in our fabric with a matching ID.  If so, and
     * its parent enclosure is "bogus", then move the links from this xbar
     * to our new xbar and delete the bogus enclosure from the fabric.
     *
     * If a matching xbar is found but is not bogus, then there may be a
     * linecard swap in progress or something.  Do nothing for now.  when
     * the other linecard goes missing, we may be able to capture his links
     * at that time.
     */
    bxp = lf_find_another_xbar_by_id(fp, xp);
    if (bxp != NULL) {
      int p;
      if (!FMS_ENC(bxp->linecard->enclosure)->is_bogus) {
	LF_ERROR(("xbar at %s:%d:%d should be bogus!",
	          bxp->linecard->enclosure->name,
		  lf_slot_display_no(bxp->linecard), bxp->xbar_no));
      }

      fms_notify(FMS_EVENT_INFO, "replacing %s",
	           bxp->linecard->enclosure->name);

      /* for each connected link, disconnect the bogus link and make
       * a new link from the new xbar
       */
      for (p=0; p<bxp->num_ports; ++p) {
	union lf_node *onp;
	int oport;

	onp = bxp->topo_ports[p];
	oport = bxp->topo_rports[p];
	if (onp != NULL) {
	  fms_remove_topo_link(LF_NODE(bxp), p);

	  /* If the node we are about to connect to is an xbar from a "bogus"
	   * switch, and we have no xcvr for the link, then the xcvr on the 
	   * bogus xbar does not really exist, so we break the link to it.
	   */
	  if (onp->ln_type == LF_NODE_XBAR
	      && FMS_ENC(LF_XBAR(onp)->linecard->enclosure)->is_bogus
	      && xp->phys_ports[p] == NULL
	      && LF_XBAR(onp)->phys_ports[oport] != NULL) {
	    struct lf_xbar *oxp;
	    struct lf_xcvr *oxcp;

	    oxp = LF_XBAR(onp);
	    oxcp = LF_XCVR(oxp->phys_ports[oport]);
	    if (oxcp->ln_type != LF_NODE_LC_XCVR) {
	      LF_ERROR(("Error: bogus phys connect is not xcvr"));
	    }

	    /* break the physical connection to the bogus xcvr */
	    lf_make_phys_link(LF_NODE(oxcp), oxp->phys_rports[oport], NULL, 0);
	    lf_make_phys_link(LF_NODE(oxp), oport, NULL, 0);
	  }
	
	  /* Now make the connections for the new xbar */
	  fms_connect_topo_nodes(LF_NODE(xp), p, onp, oport);
	  fms_add_topo_link_to_db(LF_NODE(xp), p);
	}
      }

      /* Remove this enclosure from the fabric */
      fms_remove_enclosure(bxp->linecard->enclosure);
    }
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Process all xbars in order of increasing min_nic_dist, picking
 * a best-guess match for each xbar in the reference map.
 */
static int
fms_mrf_match_xbars_by_conns(
  struct lf_fabric *fp,
  struct lf_fabric *rfp)
{
  int dist;
  int num_done;
  int rc;
  int x;

  /* Start with distance 1 xbars and work up the graph */
  dist = 1;
  do {
    num_done = 0;

    /* loop through all xbars of this length and try to match each one */
    for (x=0; x<fp->num_xbars; ++x) {
      struct lf_xbar *xp;

      xp = fp->xbars[x];

      /* If not already mated, try to figure it out by connections */
      if (FMS_XBAR(xp)->mrf_mate == NULL
	  && xp->min_nic_dist == dist) {
	  ++num_done;

	rc = fms_mrf_match_xbar_by_conns(xp);
	if (rc == -1) {
	  LF_ERROR(("Error matching xbar"));
	}
      }
    }

    ++dist;	/* move on to the next distance */
  } while (num_done > 0);

  /* Do one cleanup pass, since min_nic_dist can be off */
  for (x=0; x<fp->num_xbars; ++x) {
    struct lf_xbar *xp;

    xp = fp->xbars[x];

    /* If not already mated, try to figure it out by connections */
    if (FMS_XBAR(xp)->mrf_mate == NULL) {

      rc = fms_mrf_match_xbar_by_conns(xp);
      if (rc == -1) {
	LF_ERROR(("Error matching xbar"));
      }
    }
  }

  return 0;

 except:
  return -1;
}

/*
 * Find the most likely partner for this xbar and the appropriate
 * port offset.
 */
static int
fms_mrf_match_xbar_by_conns(
  struct lf_xbar *xp)
{
  struct lf_xbar *candidates[LF_MAX_XBAR_PORTS];
  int offsets[LF_MAX_XBAR_PORTS];
  int votes[LF_MAX_XBAR_PORTS];
  int num_cands;
  int rc;
  int p;
  int i;

  num_cands = 0;

  /* scan through each port in this xbar.
   * if something on the other end (NIC or mated xbar) */
  for (p=0; p<xp->num_ports; ++p) {
    union lf_node *conn_np;
    struct lf_xbar *cand_xp;
    int cand_offset;

    /* if nothing connected on this port, skip it */
    conn_np = xp->topo_ports[p];
    if (conn_np == NULL) continue;

    /* If mated NIC, get the mate and then find our mate candidate */
    if (conn_np->ln_type == LF_NODE_NIC
	&& FMS_NIC_N(conn_np)->mrf_mate != NULL) {
      struct lf_nic *conn_mate_nicp;

      conn_mate_nicp = FMS_NIC_N(conn_np)->mrf_mate;
      cand_xp = LF_XBAR(conn_mate_nicp->topo_ports[xp->topo_rports[p]]);
      cand_offset = p - conn_mate_nicp->topo_rports[xp->topo_rports[p]];

    } else if (conn_np->ln_type == LF_NODE_XBAR
	       && FMS_XBAR_N(conn_np)->mrf_mate != NULL) {
      struct lf_xbar *conn_mate_xp;

      conn_mate_xp = FMS_XBAR_N(conn_np)->mrf_mate;
      cand_xp = LF_XBAR(conn_mate_xp->topo_ports[xp->topo_rports[p]]);
      cand_offset = p - conn_mate_xp->topo_rports[xp->topo_rports[p]];

    } else {
      continue;
    }

    /* If cand_xp NULL, no help... */
    if (cand_xp == NULL) continue;

    /* If candidate not an xbar, that's worthy of complaint */
    if (cand_xp->ln_type != LF_NODE_XBAR) {
      LF_ERROR(("Error: candidate is not an xbar"));
    }

    /* We have a candidate - see if already registered.  If not, allocate
     * a new slot.
     */
    for (i=0; i< num_cands; ++i) {
      if (candidates[i] == cand_xp && offsets[i] == cand_offset) {
	break;
      }
    }

    /* if we ran off the end, use the next slot */
    if (i >= num_cands) {
      candidates[i] = cand_xp;
      offsets[i] = cand_offset;
      ++num_cands;
      votes[i] = 1;

    /* otherwise, count the vote */
    } else {
      ++votes[i];
    }
  }

  /* The votes are all in - find the winner */
  {
    int most_votes;
    int winner;

    most_votes = 0;
    winner = -1;
    for (i=0; i<num_cands; ++i) {
      if (votes[i] > most_votes) {
	winner = i;
	most_votes = votes[i];
      }
    }

    /* If no votes cast, we are done */
    if (most_votes <= 0) {
      return 0;
    }

    /* mate this xbar to the winner */
if (F.debug) printf("by conn: %s is really %s:%d:%d\n",
    candidates[winner]->linecard->enclosure->name,
    xp->linecard->enclosure->name,
    lf_slot_display_no(xp->linecard),
    xp->xbar_no);

    rc = fms_mrf_mate_xbars(xp, candidates[winner], offsets[winner]);
    FMS_XBAR(xp)->mrf_resolved = TRUE;
    if (rc == -1) {
      LF_ERROR(("Error assigning xbar mate"));
    }
  }

  return 0;

 except:
  return -1;
}

/*
 * Mate 2 xbars together, adjusting the reference xbar port numbers by "offset"
 */
static int
fms_mrf_mate_xbars(
  struct lf_xbar *xp,
  struct lf_xbar *rxp,
  int offset)
{
  int rc;

  /* set the mate pointers */
  FMS_XBAR(rxp)->mrf_mate = xp;
  FMS_XBAR(xp)->mrf_mate = rxp;

  /* update all the connections from this xbar with new port offset */
  rc = fms_adjust_xbar_port_nos(rxp, offset);
  if (rc == -1) {
    LF_ERROR(("Error adjusting xbar port numbers"));
  }

  return 0;

 except:
  return -1;
}

/*
 * Find any unmated xbars in the reference fabric and add them to
 * the primary fabric.  Since reference fabric xbars each have their
 * own enclosure, clone and add the enclosure to the primary fabric.
 * Copy all the connections that we can.
 *
 *   ref xp(rxp) --------------- ref node(conn_np)
 *      |                              |
 *   new xp(nxp) --make links-- existing node(conn_mate_np)
 */
static int
fms_mrf_process_unmatched_reference_xbars(
  struct lf_fabric *fp,
  struct lf_fabric *rfp)
{
  int rc;
  int x;

  /* loop through all xbars looking for unmated ones */
  for (x=0; x<rfp->num_xbars; ++x) {
    struct lf_xbar *rxp;
    struct lf_enclosure *nep;
    struct lf_xbar *nxp;
    int p;

    /* get ptr to xbar.  if mated, move on to the next one */
    rxp = rfp->xbars[x];
    if (FMS_XBAR(rxp)->mrf_mate != NULL) continue;

    /* make a copy of this enclosure */
    nep = fms_clone_enclosure(rxp->linecard->enclosure);
    if (nep == NULL) LF_ERROR(("Error cloning reference enclosure"));

    /* Add this enclosure to the fabric */
    rc = lf_add_existing_enclosure_to_fabric(fp, nep);
    if (rc == -1) {
      LF_ERROR(("Error adding enclosure to fabric"));
    }

    /* Add this to the DB fabric also - though it may leave later */
    fms_add_enclosure_to_db(nep);

    /* get to first xbar in first slot of cloned enclosure */
    nxp = LF_XBAR(nep->slots[0]->xbars[0]);
    rc = fms_mrf_mate_xbars(nxp, rxp, 0);
    if (rc == -1) LF_ERROR(("Error mating cloned xbar"));

    /* connect this new xbar to the mates of any mated reference connections */
    for (p=0; p<rxp->num_ports; ++p) {
      union lf_node *conn_np;
      union lf_node *conn_mate_np;
      int conn_port;

      /* get our connection */
      conn_np = rxp->topo_ports[p];
      if (conn_np == NULL) continue;
      conn_port = rxp->topo_rports[p];

      /* If mated NIC, get the mate */
      if (conn_np->ln_type == LF_NODE_NIC
	  && FMS_NIC_N(conn_np)->mrf_mate != NULL) {

	/* If already connected to something, remove that link */
	conn_mate_np = LF_NODE(FMS_NIC_N(conn_np)->mrf_mate);
	if (LF_NIC(conn_mate_np)->topo_ports[conn_port] != NULL) {

if (F.debug) {
printf("1 Remove link from %s to %s\n", 
  lf_node_string(conn_mate_np, conn_port),
  lf_node_string(LF_NIC(conn_mate_np)->topo_ports[conn_port], LF_NIC(conn_mate_np)->topo_rports[conn_port]));
}
	  fms_remove_topo_link(conn_mate_np, conn_port);
	}

      } else if (conn_np->ln_type == LF_NODE_XBAR
		 && FMS_XBAR_N(conn_np)->mrf_mate != NULL) {

	/* If already connected to something, remove that link */
	conn_mate_np = LF_NODE(FMS_XBAR_N(conn_np)->mrf_mate);

	if (LF_XBAR(conn_mate_np)->topo_ports[conn_port] != NULL) {
if (F.debug) {
printf("2 Remove link from %s to %s\n",
  lf_node_string(conn_mate_np, conn_port),
  lf_node_string(LF_XBAR(conn_mate_np)->topo_ports[conn_port], LF_XBAR(conn_mate_np)->topo_rports[conn_port]));
}
	  fms_remove_topo_link(conn_mate_np, conn_port);
	  /* XXX should be?  fms_note_link_removed(conn_mate_np, conn_port); */
	}

      } else {
	continue;
      }

      /* connect the mate to this xbar */
if (F.debug) {
printf("1 Add link from %s to %s\n", 
  lf_node_string(conn_mate_np, conn_port),
  lf_node_string(LF_NODE(nxp), p));
}
      fms_connect_topo_nodes(conn_mate_np, conn_port, LF_NODE(nxp), p);
      /* XXX note new link for DB fabric ? */
      fms_add_topo_link_to_db(conn_mate_np, conn_port);

    }
  }
  return 0;

 except:
  return -1;
}

/*
 * Scan all xbars in the fabric.
 * If not mated, mark all port links as "down".
 * If mated, make sure that all connections are to the mates of our mate's
 * connections.  e.g.:
 *
 *   xbar(xp) ----------- old node(old_conn_np)  new node(new_conn_np)
 *       |                                             |
 * ref xbar(mate_xp) --------------------------- ref node(mate_conn_np)
 *
 * (old node is where we *were* connected, new node is where we *are*
 * connected now)
 * If not, change this xbar to point to the mate of whatever our mate 
 * is connected to.  Error if our mate's link is unmated.
 */
static int
fms_mrf_adjust_xbar_links(
  struct lf_fabric *fp)
{
  int x;

  /* scan through all the xbars */
  for (x=0; x<fp->num_xbars; ++x) {
    struct lf_xbar *xp;
    struct lf_xbar *mate_xp;
    int p;

    xp = fp->xbars[x];
    mate_xp = FMS_XBAR(xp)->mrf_mate;

    /* If not mated, mark all ports down */
    if (mate_xp == NULL) {
      for (p=0; p<xp->num_ports; ++p) {
	/* XXX hack? - don't report internal links from un-mated xbars down */
	if (xp->topo_ports[p] != xp->phys_ports[p]) {
	  fms_mrf_link_down(LF_NODE(xp), p);
	}
      }

    /* make sure all links point to the right place */
    } else {
      for (p=0; p<xp->num_ports; ++p) {
	union lf_node *mate_conn_np;
	union lf_node *new_conn_np;
	union lf_node *old_conn_np;
	int new_conn_port;
	int old_conn_port;

	/* mate may have a more accurate port count */
	if (p >= mate_xp->num_ports) break;

	/* get pointer to a live connection */
	mate_conn_np = mate_xp->topo_ports[p];
	old_conn_np = xp->topo_ports[p];

	/* If this is non-NULL, there is a connection */
	if (mate_conn_np != NULL) {
	  new_conn_port = mate_xp->topo_rports[p];
	  old_conn_port = xp->topo_rports[p];

	  if (mate_conn_np->ln_type == LF_NODE_XBAR) {
	    new_conn_np = LF_NODE(FMS_XBAR_N(mate_conn_np)->mrf_mate);
	  } else if (mate_conn_np->ln_type == LF_NODE_NIC) {
	    new_conn_np = LF_NODE(FMS_NIC_N(mate_conn_np)->mrf_mate);
	  } else {
	    LF_ERROR(("Bad node type!"));
	  }

	  /* Mate's link must be mated! */
	  if (new_conn_np == NULL) {
	    LF_ERROR(("Mate's link is unmated!"));
	  }

	  /* xp:p meeds to get linked to new_conn_np:new_conn_port */
	  if (old_conn_np != new_conn_np || old_conn_port != new_conn_port) {

	    /* If there is an old connection, we need to unlink it. */
	    if (old_conn_np != NULL) {
if (F.debug) {
printf("3 Remove link from %s to %s\n",
lf_node_string(LF_NODE(xp), p),
lf_node_string(xp->topo_ports[p], xp->topo_rports[p]));
}
	      fms_remove_topo_link(LF_NODE(xp), p);
	    }

	    /* If new_conn_np already linked to something, unlink that, too */
	    old_conn_np = lf_follow_topo_link(new_conn_np, new_conn_port,
		                              &old_conn_port);
	    if (old_conn_np != NULL) {
if (F.debug) {
printf("3a Remove link from %s to %s\n",
lf_node_string(new_conn_np, new_conn_port),
lf_node_string(old_conn_np, old_conn_port));
}
	      fms_remove_topo_link(new_conn_np, new_conn_port);
	    }


	    /* Now add the new link */
if (F.debug) {
printf("2 Add link from %s to %s\n",
lf_node_string(LF_NODE(xp), p),
lf_node_string(new_conn_np, new_conn_port));
}
	    fms_connect_topo_nodes(LF_NODE(xp), p, new_conn_np, new_conn_port);
	    fms_add_topo_link_to_db(LF_NODE(xp), p);
	    fms_mrf_link_up(LF_NODE(xp), p);

	  /* connected to the same place - if link if down, mark it up now */
	  } else {
	    fms_mrf_link_up(LF_NODE(xp), p);
	  }

	/* mate_conn_np == NULL means no link - mark it down */
	} else if (old_conn_np != NULL) {
	  fms_mrf_link_down(LF_NODE(xp), p);
	}
      }
    }
  }
  return 0;

 except:
  return -1;
}

/*
 * Mark a link down and raise an alert if necessary
 */
static void
fms_mrf_link_down(
  union lf_node *np,
  int port)
{
  struct fms_link *linkp;

  linkp = fms_get_node_link(np, port);

  /* we don't want to make it "down" if, e.g., in MAINT mode */
  if (lf_get_node_link_state(np, port) == LF_LINK_STATE_UP) {
    lf_set_node_link_state(np, port, LF_LINK_STATE_DOWN);

    /* If this is an xbar, raise appropriate alert */
    if (np->ln_type == LF_NODE_XBAR && linkp->fl_ref_cnt > 1) {
      fms_switch_alert_xbar_link_down(LF_XBAR(np), port);

    /* If NIC and other end unknown, use misc nic_down alert */
    } else if (np->ln_type == LF_NODE_NIC && linkp->fl_ref_cnt < 2) {
      fms_fma_alert_nic_port_down(LF_NIC(np), port);
    }
  }
}

/*
 * Mark a link up, but don't do anything else about it
 */
static void
fms_mrf_link_up(
  union lf_node *np,
  int port)
{
  struct fms_link *linkp;
  enum lf_link_state old_state;

  /* we don't want to make it "up" if, e.g., in MAINT mode */
  old_state = lf_get_node_link_state(np, port);
  if (old_state == LF_LINK_STATE_DOWN || old_state == LF_LINK_STATE_UNKNOWN) {

    /* mark the link up */
    lf_set_node_link_state(np, port, LF_LINK_STATE_UP);

    /* rescind any alerts about it being down */
    linkp = fms_get_node_link(np, port);
    fms_switch_alert_link_up(linkp);

    if (np->ln_type == LF_NODE_NIC) {
      fms_fma_alert_nic_port_up(LF_NIC(np), port);
    }
  }
}

/*
 * Scan all NICs in the fabric.
 * If not mated, mark all port links as "down".
 * If mated, make sure that all connections are to the mates of our mate's
 * connections.  e.g.:
 *
 *   NIC(nicp) ----------- old node(old_conn_np)  new node(new_conn_np)
 *       |                                                |
 * ref NIC(mate_nicp) --------------------------- ref node(mate_conn_np)
 *
 * (old node is where we *were* connected, new node is where we *are*
 * connected now)
 * If not, change this NIC to point to the mate of whatever our mate 
 * is connected to.  Error if our mate's link (mate_conn_np) is unmated.
 */
static int
fms_mrf_adjust_nic_links(
  struct lf_fabric *fp)
{
  int h;

  for (h=0; h<fp->num_hosts; ++h) {
    struct lf_host *hp;
    int n;

    hp = fp->hosts[h];
    for (n=0; n<hp->num_nics; ++n) {
      struct lf_nic *nicp;
      struct lf_nic *mate_nicp;
      int p;

      /* get to NIC and its mate */
      nicp = hp->nics[n];
      mate_nicp = FMS_NIC(nicp)->mrf_mate;

      /* If not mated, mark all ports down */
      if (mate_nicp == NULL) {
	fms_notify(FMS_EVENT_DEBUG, "%s:n%d is unmated!", hp->hostname, n);

	for (p=0; p<nicp->num_ports; ++p) {
	  fms_mrf_link_down(LF_NODE(nicp), p);
	  fms_notify(FMS_EVENT_DEBUG, "Port %d of %s is down",
		     p, nicp->host->hostname);
	}

      /* make sure all links point to the right place */
      } else {
	for (p=0; p<nicp->num_ports; ++p) {
	  union lf_node *mate_conn_np;
	  union lf_node *new_conn_np;
	  union lf_node *old_conn_np;
	  int new_conn_port;
	  int old_conn_port;

	  /* get pointers to new and old connections */
	  mate_conn_np = mate_nicp->topo_ports[p];
	  old_conn_np = nicp->topo_ports[p];

	  /* If this is non-NULL, there is a connection */
	  if (mate_conn_np != NULL) {
	    new_conn_port = mate_nicp->topo_rports[p];
	    old_conn_port = nicp->topo_rports[p];

	    if (mate_conn_np->ln_type == LF_NODE_XBAR) {
	      new_conn_np = LF_NODE(FMS_XBAR_N(mate_conn_np)->mrf_mate);
	    } else if (mate_conn_np->ln_type == LF_NODE_NIC) {
	      new_conn_np = LF_NODE(FMS_NIC_N(mate_conn_np)->mrf_mate);
	    } else {
	      LF_ERROR(("Bad node type!"));
	    }

	    /* Mate's link must be mated! */
	    if (new_conn_np == NULL) {
	      LF_ERROR(("Mate's link is unmated!"));
	    }

	    /* nicp:p meeds to get linked to new_conn_np:new_conn_port */
	    if (old_conn_np != new_conn_np || old_conn_port != new_conn_port) {

	      /* If there is an old connection, we need to unlink it. */
	      if (old_conn_np != NULL) {
if (F.debug) {
printf("4 Remove link from %s to %s\n",
lf_node_string(LF_NODE(nicp), p),
lf_node_string(nicp->topo_ports[p], nicp->topo_rports[p]));
}
		fms_remove_topo_link(LF_NODE(nicp), p);
	      }

	      /* Now add the new link */
if (F.debug) {
printf("3 Add link from %s to %s\n",
lf_node_string(LF_NODE(nicp), p),
lf_node_string(new_conn_np, new_conn_port));
}
	      fms_connect_topo_nodes(LF_NODE(nicp), p,
		                     new_conn_np, new_conn_port);
	      fms_add_topo_link_to_db(LF_NODE(nicp), p);
	    }

	  /* mate_conn_np == NULL means no link - mark it down */
	  } else if (old_conn_np != NULL) {
	    fms_mrf_link_down(LF_NODE(nicp), p);
	    fms_notify(FMS_EVENT_DEBUG, "Port %d of %s is down",
		       p, nicp->host->hostname);
	  }
	}
      }
    }
  }
  return 0;

 except:
  return -1;
}

/*
 * Perform fabric validation on our newly modified fabric
 */
static int
fms_mrf_validate_fabric(
  struct lf_fabric *fp)
{
  return 0;
}
